//=============================================================================
//
// file :        ClassAttribute.h
//
// description :    Include file for the Tango attribute.
//            Three classes are declared in this file :
//
//            AttrProperty : A helper class. This class stored a couple
//                       property name, property value as strings
//             ClassAttribute : A class for each attribute with all
//                     its properties defined at the class
//                     level. The properties are still stored
//                     as string
//            MultiClassAttribute : A class to manage all the class
//                          level definition of attributes.
//                          There is one instance of this class
//                          for each tango device pattern
//
// project :        TANGO
//
// author(s) :        E.Taurel
//
// Copyright (C) :      2004,2005,2006,2007,2008,2009,2010,2011,2012,2013,2014,2015
//                        European Synchrotron Radiation Facility
//                      BP 220, Grenoble 38043
//                      FRANCE
//
// This file is part of Tango.
//
// Tango is free software: you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Tango is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public License
// along with Tango.  If not, see <http://www.gnu.org/licenses/>.
//
//
//=============================================================================

#ifndef _CLASS_ATTRIBUTE_H
#define _CLASS_ATTRIBUTE_H

#include <string>
#include <vector>

namespace Tango
{
class Attr;

//=============================================================================
//
//            The AttrProperty class
//
//
// description :    This class is used to store a pair of attribute name,
//            attribute value. Both name and value are stored as
//            string
//
//=============================================================================

class AttrProperty
{
  public:
    AttrProperty(const std::string &name, const std::string &value);
    AttrProperty(const std::string &name, long value);
    ~AttrProperty();

    AttrProperty(const AttrProperty &);
    AttrProperty &operator=(const AttrProperty &);

    AttrProperty(AttrProperty &&);
    AttrProperty &operator=(AttrProperty &&);

    std::string &get_value()
    {
        return attr_value;
    }

    std::string const &get_value() const
    {
        return attr_value;
    }

    long get_lg_value() const
    {
        return attr_lg;
    }

    std::string &get_name()
    {
        return attr_name;
    }

    std::string const &get_name() const
    {
        return attr_name;
    }

    void convert(const char *);

  private:
    std::string attr_name;
    std::string attr_value;
    long attr_lg;
};

//=============================================================================
//
//            The MultiClassAttribute class
//
//
// description :    This class is a holder for all the ClassAttribute
//            instance. There is one instance of this class for each
//            Tango device pattern implementation. This instance is
//            stored in the DeviceClass object of the pattern
//
//=============================================================================

class MultiClassAttribute
{
  public:
    MultiClassAttribute();
    ~MultiClassAttribute();

    void init_class_attribute(const std::string &class_name, long base = 0);

    std::vector<Tango::Attr *> &get_attr_list()
    {
        return attr_list;
    }

    Attr &get_attr(const std::string &attr_name);
    void remove_attr(const std::string &, const std::string &);

  protected:
    std::vector<Tango::Attr *> attr_list;
};

} // namespace Tango

#endif // _CLASS_ATTRIBUTE_H
